/**
* \file: device.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <string.h>

#include "model/device.h"
#include "model/device_metadata.h"
#include "model/partition_list.h"
#include "model/model_internal.h"

#include "app_iface/app_iface_intern.h"

#include "utils/automounter_types_internal.h"

static void device_signal_state_change(device_t *device, device_state_t old_state);

device_t *device_create_new(const char* device_id, device_metadata_t *metadata, devicehandler_vtable_t *vtable)
{   
	device_t *device;
	device = malloc(sizeof(device_t));
	if (device!=NULL)
	{
	    memset(device,0,sizeof(device_t));
		device->id=strdup(device_id);
		if (device->id!=NULL)
		{
			device->state = DEVICE_DETECTED;
			device->detected_partition_cnt=0;
			device->expected_partition_cnt=0;
			device->metadata = metadata;
			device->device_vtable = vtable;

			device->partition_list_head.device = device;
		}
		else
		{
			device_free(device);
			device=NULL;
		}
	}
	return device;
}

void device_free(device_t *device)
{
	partition_t *partition;

	//delete partition structures that might be still in the list
	partition = device->partition_list_head.next_partition;

	//cleanly empty the list
	while (partition != NULL)
	{
		partition_t *next_par=partition->next_partition;
		partition_free(partition);
		partition=next_par;
	}

	free(device->id);

	device_metadata_free(device);

	free(device);
}

const char* device_get_id(device_t *device)
{
	return device->id;
}

const char *device_get_device_type(device_t *device)
{
	return device_metadata_get_device_type(device);
}

const char *device_get_device_identifier(device_t *device)
{
	return device_metadata_get_device_identifier(device);
}

device_metadata_t *device_get_metadata(device_t *device)
{
	return device->metadata;
}

const char *device_get_handler_id(device_t *device)
{
	return device->device_vtable->device_handler_id;
}


void device_set_state_change_monitor(device_t *device,
                    device_state_change_callback_t callback_func, void *callback_data)
{
    device->state_change_callback_func=callback_func;
    device->state_change_callback_data=callback_data;
}

void device_remove_state_change_monitor(device_t *device)
{
    device->state_change_callback_func=NULL;
    device->state_change_callback_data=NULL;
}



device_state_t device_get_state(device_t *device)
{
    return device->state;
}

const char* device_get_state_str(device_state_t state)
{
	return automounter_get_device_state_string(state);
}

void device_set_state_automounting(device_t *device, int expected_partition_cnt)
{
    device_state_t old_state;
    old_state=device->state;

	device->expected_partition_cnt=expected_partition_cnt;
	device->state=DEVICE_AUTOMOUNTING;
	app_iface_signal_device_automounting(device);
    device_signal_state_change(device,old_state);
}

void device_set_state_automounted(device_t *device)
{
    device_state_t old_state;
    old_state=device->state;

    device->state=DEVICE_AUTOMOUNTED;
	app_iface_signal_device_automounted(device);
    device_signal_state_change(device,old_state);
}

void device_set_state_unmounting(device_t *device)
{
    device_state_t old_state;
    old_state=device->state;

    device->state=DEVICE_UNMOUNTING;
	app_iface_signal_device_unmounting(device);
    device_signal_state_change(device,old_state);
}

void device_set_state_unmounted(device_t *device)
{
    device_state_t old_state;
    old_state=device->state;

    device->state=DEVICE_UNMOUNTED;
	app_iface_signal_device_unmounted(device);
    device_signal_state_change(device,old_state);
}

void device_set_state_invalid(device_t *device)
{
    device_state_t old_state;
    old_state=device->state;

    device->state=DEVICE_INVALID;
	device->expected_partition_cnt=0;
	app_iface_signal_device_invalid(device);
    device_signal_state_change(device,old_state);
}

void device_set_state_nomedia(device_t *device)
{
    device_state_t old_state;
    old_state=device->state;

    device->state=DEVICE_NOMEDIA;
	device->expected_partition_cnt=0;
	app_iface_signal_device_nomedia(device);
    device_signal_state_change(device,old_state);
}

int device_get_detected_partition_cnt(device_t *device)
{
	return device->detected_partition_cnt;
}

int device_get_expected_partition_cnt(device_t *device)
{
	return device->expected_partition_cnt;
}

int device_get_working_partitions_cnt(device_t *device)
{
	int cnt=0;
	partition_iterator_t itr;
	partition_t *partition;
	partition_state_t state;

	partition=device_first_partition(device,&itr);
	while(partition!=NULL)
	{
		state=partition_get_state(partition);
		if (state==PARTITION_MOUNTING || state==PARTITION_UNMOUNTING ||
				state==PARTITION_REMOUNTING)
			cnt++;
		partition=device_next_partition(&itr);
	}
	return cnt;
}

int device_get_mounted_partitions_cnt(device_t *device)
{
	int cnt=0;
	partition_iterator_t itr;
	partition_t *partition;

	partition=device_first_mounted_partition(device,&itr);
	while(partition!=NULL)
	{
		cnt++;
		partition=device_next_mounted_partition(&itr);
	}
	return cnt;
}

static void device_signal_state_change(device_t *device, device_state_t old_state)
{
  if (device->state_change_callback_func!=NULL)
  {
    device->state_change_callback_func(device, old_state, device->state_change_callback_data);
    device_remove_state_change_monitor(device);
  }
}
